
/* Copyright (C) 2002-2018 RealVNC Ltd. All Rights Reserved. */

#ifndef UUID_7981a208_ccb4_4d99_bf84_e3cb0571b7a8
#define UUID_7981a208_ccb4_4d99_bf84_e3cb0571b7a8

/**
 * \file vncsbptypes.h
 *
 * Contains enumerations and types used by SBP.  The application should
 * normally include vnccdbsdk.h rather than including this file directly.
 *
 * \see vnccdbsdk.h, vncsbp.h
 */

#include "vncint.h"


#ifdef __cplusplus
extern "C"
{
#endif


/** 
 * \brief SBP command types - From SBP spec 3.3 (v1.1.1) 
 * 
 * Valid command types are in the range 0xb1 - 0xb9. The range 0xba - 0xbf
 * is reserved for the next update. For v1.1, the SBP Source must return
 * ::VNCSBPProtocolErrorFeatureNotSupported if a command in this range is
 * received.
 */
typedef enum
{
  VNCSBPCommandTypeInvalidLow                   = 0xb0,

  /**
   * \brief Reads an object value once.
   */
  VNCSBPCommandTypeGet                          = 0xb1,
  
  /**
   * \brief Write an object value.
   */
  VNCSBPCommandTypeSet                          = 0xb2,
  
  /**
   * \brief Get multiple notifications for an object value. 
   * 
   * Object data is sent later depending on subscription type and interval. 
   */
  VNCSBPCommandTypeSubscribe                    = 0xb3,
  
  /**
   * \brief Cancel the currently active command with the specified type.
   */
  VNCSBPCommandTypeCancel                       = 0xb4,
  
  /**
   * \brief Request the SBP source to send an AliveResponse.
   */
  VNCSBPCommandTypeAliveRequest                 = 0xb5,
  
  /**
   * \brief Reply for AliveRequest.
   */
  VNCSBPCommandTypeAliveResponse                = 0xb6,
  
  /**
   * \brief Authentication challenge.
   * 
   * This is used when an object which requires authentication to be 
   * read from / written to is accessed. The data passed is defined by each
   * authentication method. In v1.1, the only authentication method 
   * supported is ::VNCSBPAuthenticationMethodServiceSpecific.
   */
  VNCSBPCommandTypeAuthenticationChallenge      = 0xb7,
  
  /**
   * \brief Authentication response.
   * 
   * Reply to an authentication challenge. For v1.1, this must return
   * ::VNCSBPProtocolErrorFeatureNotSupported except for service specific
   * authentication.
   */
  VNCSBPCommandTypeAuthenticationResponse       = 0xb8,
  
  /**
   * \brief Response.
   * 
   * This is a response for Get, Subscribe, Cancel and AuthenticationResponse.
   */
  VNCSBPCommandTypeResponse                     = 0xb9,

  VNCSBPCommandTypeInvalidHigh                  = 0xba,

} VNCSBPCommandType;


#define VNCSBPIsCommandTypeValid(command) \
  (command > VNCSBPCommandTypeInvalidLow && \
   command < VNCSBPCommandTypeInvalidHigh)


/** \brief SBP subscription types - From SBP spec 3.3 (v1.1.1) */
typedef enum
{
  /** 
   * \brief Send updates at regular intervals
   * 
   * The interval is specified in milliseconds. If it is outside the
   * range the SBP Source supports, it should return error
   * ::VNCSBPProtocolErrorWrongSubscriptionInterval.
   */
  VNCSBPSubscriptionTypeRegularInterval         = 0x0,
  
  /**
   * \brief Send update when there is a change.
   * 
   * When there are too many frequent changes, the SBP Source can device
   * to either drop some updates, or combine multiple updates into a single
   * one if the data structure allows it.
   */
  VNCSBPSubscriptionTypeOnChange                = 0x1,
  
  /**
   * \brief Automatic
   * 
   * It is up to the SBP Source to choose regular intervals or on change.
   * The SBP Source can choose the optimal notification mechanism for the 
   * requested object.
   */
  VNCSBPSubscriptionTypeAutomatic               = 0x2,

  VNCSBPSubscriptionTypeInvalidHigh             = 0x3

} VNCSBPSubscriptionType;

/** \brief Maximum interval for RegularInterval subscriptions. */
static const vnc_uint32_t VNCSBPMaximumSubscriptionInterval = 0x00ffffff;


/** \brief SBP Authentication methods - from SBP spec 3.3 (v1.1.1) */
typedef enum
{
  /** 
   * \brief Service uses a service-specific method of authentication.
   * 
   * In MirrorLink version 1.1, this is the only supported authentication 
   * method.
   */
  VNCSBPAuthenticationMethodServiceSpecific     = 0x80000000
} VNCSBPAuthenticationMethod;


/** 
 * \brief SBP error codes - from SBP spec 3.6.4 (v1.1.1) 
 * 
 * Error codes in the range 0x00000001 to 0x0fffffff are unrecoverable and
 * will result in the service being stopped / reset.
 * 
 * Error codes in the range 0x10000000 to 0x3fffffff are recoverable.
 * 
 * Error codes in the range 0x40000000 to 0x4fffffff are reserved for 
 * service-specific error codes.
 * 
 * As new error codes may be added in the future, the range of the error 
 * code should be checked using the macro ::VNCSBPIsErrorRecoverable first
 * rather than checking individual error codes.
 */
typedef enum
{
  /** \brief No error. */
  VNCSBPProtocolErrorNone                         = 0x0,
  
  /** 
   * \brief Unknown data type.
   * 
   * This error is unrecoverable.
   */
  VNCSBPProtocolErrorUnknownDataType              = 0x1,
  
  /** 
   * \brief Wrong END. 
   * 
   * Command does not terminate with END/END_C, or END/END_C is found in 
   * the wrong place. 
   * 
   * This error is unrecoverable. 
   */
  VNCSBPProtocolErrorWrongEND                     = 0x2,
  
  /** 
   * \brief Wrong element data type. 
   * 
   * This is the case when the SBP Source has set the wrong element data 
   * type for an array.
   * 
   * This error is unrecoverable. 
   */
  VNCSBPProtocolErrorWrongElementType             = 0x3,
  
  /**
   * \brief UID and type do not match.
   * 
   * The type bound with the UID does not match with the type actually 
   * transferred.
   * 
   * This error is unrecoverable.
   */
  VNCSBPProtocolErrorUIDTypeMismatch              = 0x4,
  
  /**
   * \brief Other unrecoverable error.
   * 
   * This may be due to an implementation-specific reason such as a memory
   * allocation failure.
   */
  VNCSBPProtocolErrorGenericUnrecoverable         = 0x01000000,
  
  /** 
   * \brief Continue.
   * 
   * SBP Source needs more time to process the request. This is the error
   * code for the Response-Continue message.
   */
  VNCSBPProtocolErrorContinue                     = 0x10000000,
  
  /** 
   * \brief Unknown UID. 
   * 
   * Service does not recognise this object UID.
   */
  VNCSBPProtocolErrorUnknownUID                   = 0x10000001,
  
  /** 
   * \brief Feature not supported.
   * 
   * This service does not support the requested feature.
   */
  VNCSBPProtocolErrorFeatureNotSupported          = 0x10000002,
  
  /** 
   * \brief Wrong subscription interval.
   * 
   * This service does not support the requested subscription interval.
   */
  VNCSBPProtocolErrorWrongSubscriptionInterval    = 0x10000003,
  
  /** 
   * \brief Wrong subscription type.
   * 
   * This service does not support the requested subscription type.
   */
  VNCSBPProtocolErrorWrongSubscriptionType        = 0x10000004,
  
  /** 
   * \brief Missing mandatory data.
   */
  VNCSBPProtocolErrorMissingMandatoryData         = 0x10000005,
  
  /** 
   * \brief The requested data is currently unavailable.
   */
  VNCSBPProtocolErrorNotAvailable                 = 0x10000006,

  /** 
   * \brief Authentication failed.
   */
  VNCSBPProtocolErrorAuthenticationFailed         = 0x10000007,

  /** 
   * \brief Command already pending.
   * 
   * The same command has been sent again before the previous one has
   * completed. 
   */
  VNCSBPProtocolErrorCommandAlreadyPending        = 0x10000008,

  /** 
   * \brief Command not pending.
   * 
   * A cancel command has been send for a command which is not pending. 
   */
  VNCSBPProtocolErrorCommandNotPending            = 0x10000009,
  
  /**
   * \brief No more session.
   * 
   * SBP Source cannot support new commands until currently active commands
   * have completed, due to resource limitations or other reasons.
   */
  VNCSBPProtocolErrorNoMoreSession                = 0x1000000a,
  
  /**
   * \brief Command successfully cancelled. 
   *
   * When a cancel request is successful, this error code should be returned.
   * 
   * Note that the OK response for the cancel command indicates only that 
   * the cancel command was successfully completed, not that the service 
   * processed the cancellation.
   */
  VNCSBPProtocolErrorCommandCancelled             = 0x1000000b,
  
  /**
   * \brief Write not allowed.
   * 
   * This object is not writable. This is the error the SBP Source must 
   * return when a Set command is sent to an object which does not support
   * writing.
   */
  VNCSBPProtocolErrorWriteNotAllowed              = 0x1000000c,
  
  /**
   * \brief Unknown command.
   */
  VNCSBPProtocolErrorUnknownCommand               = 0x1000000d,

  /**
   * \brief Object not available for writing.
   *
   * This object is currently not available for writing.
   */
  VNCSBPProtocolErrorObjectNotAvailableForWriting = 0x1000000e,

  /**
   * \brief Other recoverable error.
   * 
   * This may be due to an implementation-specific reason.
   */
  VNCSBPProtocolErrorGenericRecoverable           = 0x11000000,

  VNCSBPProtocolErrorServiceSpecificBase          = 0x40000000

} VNCSBPProtocolError;


/** \brief Lower limit of recoverable error code range. */
static const VNCSBPProtocolError VNCSBPLowestRecoverableError
        = VNCSBPProtocolErrorContinue;


/** 
 * \brief Check if the specified error code represents a recoverable error.
 *
 * \param[in]   code            Error code to check.
 *
 * \return      true iff the specified code is in the recoverable range.
 */
#define VNCSBPIsErrorRecoverable(code) \
  ((code) >= VNCSBPLowestRecoverableError)


#ifdef __cplusplus
}
#endif

#endif /* ifndef UUID_7981a208_ccb4_4d99_bf84_e3cb0571b7a8 */

